/*
  LadderDAC8 - Potentiometer Feedback (OLED Output)
  
  This example demonstrates the basic DC voltage generation capability of the 
  LadderDAC8 shield and tests the accuracy of the R-2R resistor network.
    
  Uses Adafruit's SSD1306 library for OLED operation.
  Place jumper to A0-DAC for feedback reading.

  Library: Mete Hoca LadderDAC 8
*/

#include <Wire.h>
#include <Adafruit_SSD1306.h>
#include <LadderDAC8.h>

Adafruit_SSD1306 display(128, 32, &Wire, -1);
LadderDAC8 dac;

float graphData[60];
int graphIndex = 0;

void setup() {
  Wire.begin(); 
  Wire.setClock(100000);
  display.begin(SSD1306_SWITCHCAPVCC, 0x3C);
  dac.begin();
  dac.setReference(4.98);
  display.clearDisplay();
  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);
  for (int i = 0; i < 60; i++) graphData[i] = 0;
}

void loop() {
  int potValue = dac.readPotRaw();
  byte dacValue = dac.readPot();
  dac.write(dacValue);

  graphData[graphIndex] = dac.readFeedbackVoltage();
  graphIndex = (graphIndex + 1) % 60;
  display.clearDisplay();

  display.setCursor(0, 0);
  display.println("POT :" + String("     ").substring(String(potValue).length()) + potValue);
  display.println("DAC :" + String("     ").substring(String(dacValue).length()) + dacValue);
  display.print("vDAC:"); display.print(dac.getDacVoltage(), 3); display.println("V");
  display.print("vFB :"); display.print(dac.readFeedbackVoltage(), 3); display.println("V");

  int graphHeight = 32;
  for (int i = 0; i < 59; i++) {
    int idx1 = (graphIndex + i) % 60;
    int idx2 = (graphIndex + i + 1) % 60;
    int y1 = graphHeight - mapmap(graphData[idx1] * 100, 0, (int)(dac.getReference() * 100), 1, graphHeight - 1);
    int y2 = graphHeight - mapmap(graphData[idx2] * 100, 0, (int)(dac.getReference() * 100), 1, graphHeight - 1);
    display.drawLine(68 + i, y1, 68 + i + 1, y2, SSD1306_WHITE);
  }

  display.display();
  delay(16);
}

long mapmap(long x, long in_min, long in_max, long out_min, long out_max) {
  long rangeIn = in_max - in_min;
  if (rangeIn == 0) return out_min;
  if (in_min < in_max) { x = constrain(x, in_min, in_max); }
    else { x = constrain(x, in_max, in_min); }
  double slope = double(x - in_min) / double(rangeIn);
  double result = slope * double(out_max - out_min) + double(out_min);
  return lround(result);
}

double mapmap(double x, double in_min, double in_max, double out_min, double out_max) {
  double rangeIn = in_max - in_min;
  if (rangeIn == 0) return out_min;
  if (in_min < in_max) { x = constrain(x, in_min, in_max); }
    else { x = constrain(x, in_max, in_min); }
  return ( (x - in_min) / rangeIn ) * (out_max - out_min) + out_min;
}