/*
  LadderDAC8 - Guided Tour
  
  Hello! This sketch is a "Tour Guide" for the LadderDAC8 library.
  We will test every function in the library one by one.
  It also includes the final optimized code for all 4 waveforms.
  
  Open the Serial Monitor (115200 baud) to watch the test steps!

  Library: Mete Hoca LadderDAC 8
*/

#include <LadderDAC8.h>

// Create our DAC object.
LadderDAC8 dac;

void setup() {
  // Start the communication with the computer
  Serial.begin(115200);
  // Wait a moment for the Serial Monitor to open (good for Uno R4)
  while (!Serial);
  delay(500);
  Serial.println("\n--- Welcome to the LadderDAC8 Guided Tour! ---\n");

  // 1. START THE DAC
  // This command prepares the pins on your Arduino.
  dac.begin(); 
  Serial.println("Step 1: Library started with dac.begin().");
}

void loop() {
  Serial.println("\n==================================================");
  Serial.println("Starting a new test cycle...");

  // ================================================================
  // PART 1: VOLTAGE SETTINGS
  // ================================================================
  
  // Set the reference voltage (5V) for accurate calculations.
  dac.setReference(5.00);
  float currentRef = dac.getReference();
  Serial.print("Reference Voltage set to: ");
  Serial.print(currentRef);
  Serial.println(" Volts");
  
  delay(3000);

  // ================================================================
  // PART 2: SIMPLE WRITING (0 to 255)
  // ================================================================
  
  // dac.write(value) -> Sends a raw number between 0 and 255.
  dac.write(127); 
  Serial.println("Testing write(127) -> Half power (approx 2.5V)");
  delay(3000);

  // dac.clear() -> Quickly sets output to 0 Volts.
  dac.clear();
  Serial.println("Testing clear() -> Output is now 0V");
  delay(3000);

  // dac.full() -> Quickly sets output to Max Voltage.
  dac.full();
  Serial.println("Testing full() -> Output is now Max Voltage");
  delay(3000);

  // ================================================================
  // PART 3: PLAYING WITH BITS (Digital Switches)
  // ================================================================
  
  dac.clear();
  // Turn ON the smallest bit/switch (Bit 0)
  dac.writeBit(0, 1);
  Serial.println("Testing writeBit(0, 1) -> Smallest switch is ON");
  delay(3000);
  
  // Turn ON the largest bit/switch (Bit 7)
  dac.writeBit(7, 1);
  Serial.println("Testing writeBit(7, 1) -> Largest switch is ON");
  delay(3000);

  // ================================================================
  // PART 4: SMART VOLTAGE (Let the code do the math)
  // ================================================================
  
  // Request a specific voltage (e.g., 2.5 Volts) directly.
  float targetVolts = 2.5;
  dac.writeVoltage(targetVolts);
  
  // Check the voltage calculated and output by the library.
  float mathResult = dac.getDacVoltage();
  
  Serial.print("I requested: ");
  Serial.print(targetVolts);
  Serial.print("V -> Library outputting: ");
  Serial.print(mathResult);
  Serial.println("V");
  
  delay(3000);

  // ================================================================
  // PART 5: WAVEFORM GENERATION (1 Hz - 5 Seconds each)
  // ================================================================
  
  float waveFreq = 1.0; // 1 Hz Frequency

  // --- SINE WAVE ---
    dac.waveSine(waveFreq);
  Serial.println("Starting 1 Hz Sine Wave. Waiting 5 seconds...");
  delay(5000); 
  
  // --- SQUARE WAVE ---
  dac.waveSquare(waveFreq);
  Serial.println("Starting 1 Hz Square Wave. Waiting 5 seconds...");
  delay(5000);

  // --- TRIANGLE WAVE ---
  dac.waveTriangle(waveFreq);
  Serial.println("Starting 1 Hz Triangle Wave. Waiting 5 seconds...");
  delay(5000);

  // --- SAWTOOTH WAVE ---
  dac.waveSawtooth(waveFreq);
  Serial.println("Starting 1 Hz Sawtooth Wave. Waiting 5 seconds...");
  delay(5000);

  dac.waveStop(); // Stop the Timer/Wave Generator

  // ================================================================
  // PART 6: SENSORS (Potentiometer & Feedback)
  // ================================================================
  
  dac.clear(); // Clear DAC output
  
  // Which pins are configured?
  Serial.print("\nPOT Pin (A2): A"); Serial.print(dac.getPotPin() - 14);
  Serial.print(" | Feedback Pin (A0): A"); Serial.println(dac.getFeedbackPin() - 14);

  // --- Reading the Knob (Potentiometer) ---
  int potRaw = dac.readPotRaw();
  int potMapped = dac.readPot();

  Serial.print("Knob Position -> Raw (0-1023): ");
  Serial.print(potRaw);
  Serial.print(" | Mapped (0-255): ");
  Serial.println(potMapped);

  // --- Reading the Output Voltage (Feedback) ---
  float actualVoltage = dac.readFeedbackVoltage();
  int feedbackRaw = dac.readFeedbackRaw();
  
  Serial.print("Actual Output -> Raw (0-1023): ");
  Serial.print(feedbackRaw);
  Serial.print(" | Voltage: ");
  Serial.print(actualVoltage);
  Serial.println(" V");

  Serial.println("\nGuided tour complete! Restarting in 3 seconds...");
  delay(3000);
}