#include "LadderDAC8.h"

#if defined(ARDUINO_ARCH_RENESAS)
    #include <FspTimer.h> 
#endif

static const uint8_t dacPins[8] = {3,4,5,6,7,8,9,10};
static const uint8_t potPin = A2;
static const uint8_t feedbackPin = A0;

static const uint8_t sineTable[256] = {
  128, 131, 134, 137, 140, 143, 146, 149, 152, 155, 158, 162, 165, 167, 170, 173,
  176, 179, 182, 185, 188, 190, 193, 196, 198, 201, 203, 206, 208, 211, 213, 215,
  218, 220, 222, 224, 226, 228, 230, 232, 234, 235, 237, 238, 240, 241, 243, 244,
  245, 246, 248, 249, 250, 250, 251, 252, 253, 253, 254, 254, 254, 255, 255, 255,
  255, 255, 255, 255, 254, 254, 254, 253, 253, 252, 251, 250, 250, 249, 248, 246,
  245, 244, 243, 241, 240, 238, 237, 235, 234, 232, 230, 228, 226, 224, 222, 220,
  218, 215, 213, 211, 208, 206, 203, 201, 198, 196, 193, 190, 188, 185, 182, 179,
  176, 173, 170, 167, 165, 162, 158, 155, 152, 149, 146, 143, 140, 137, 134, 131,
  128, 124, 121, 118, 115, 112, 109, 106, 103, 100, 97, 93, 90, 88, 85, 82,
  79, 76, 73, 70, 67, 65, 62, 59, 57, 54, 52, 49, 47, 44, 42, 40,
  37, 35, 33, 31, 29, 27, 25, 23, 21, 20, 18, 17, 15, 14, 12, 11,
  10, 9, 7, 6, 5, 5, 4, 3, 2, 2, 1, 1, 1, 0, 0, 0,
  0, 0, 0, 0, 1, 1, 1, 2, 2, 3, 4, 5, 5, 6, 7, 9,
  10, 11, 12, 14, 15, 17, 18, 20, 21, 23, 25, 27, 29, 31, 33, 35,
  37, 40, 42, 44, 47, 49, 52, 54, 57, 59, 62, 65, 67, 70, 73, 76,
  79, 82, 85, 88, 90, 93, 97, 100, 103, 106, 109, 112, 115, 118, 121, 124
};

#if defined(__AVR__)
    static uint8_t _portD_Map[256]; 
    static uint8_t _portB_Map[256];
    static bool _timerRunning = false;
#elif defined(ARDUINO_ARCH_RENESAS)
    static uint16_t _r4_Port1_Map[256];
    static uint16_t _r4_Port3_Map[256];
    static int8_t _r4TimerIndex = -1;
#endif

LadderDAC8* _timerDAC = nullptr;
volatile uint8_t _sineIdx = 0;
volatile uint8_t _increment = 1; 
uint8_t _currentWaveMode = 255; 

#if defined(ARDUINO_ARCH_RENESAS)
    FspTimer _r4Timer; 
#endif

void _updatePortMaps(uint8_t mode) {
    if (_currentWaveMode == mode) return;
    _currentWaveMode = mode;

    for (int i = 0; i < 256; i++) {
        uint8_t val = 0;
        if (mode == 0) val = sineTable[i];
        else if (mode == 1) val = (i < 128) ? 255 : 0;
        else if (mode == 2) val = (i < 128) ? i * 2 : (255 - i) * 2;
        else if (mode == 3) val = i;

        #if defined(__AVR__)
            _portD_Map[i] = (val << 3) & 0xF8;
            _portB_Map[i] = (val >> 5) & 0x07;
        #elif defined(ARDUINO_MINIMA) || defined(ARDUINO_NANOR4)
            _r4_Port1_Map[i] = ((val & 0x01) << 4) | ((val & 0x02) << 2) | 
                               (val & 0x04)        | ((val & 0x08) << 3) | 
                               ((val & 0x10) << 3) | ((val & 0x80) << 5);
            _r4_Port3_Map[i] = ((val & 0x20) >> 1) | ((val & 0x40) >> 3);
        #elif defined(ARDUINO_UNOWIFIR4)
            _r4_Port1_Map[i] = ((val & 0x01) << 5) | ((val & 0x02) << 5) | 
                               ((val & 0x04) << 5) | ((val & 0x08) << 8) | 
                               ((val & 0x10) << 8) | ((val & 0x80) >> 4);
            _r4_Port3_Map[i] = ((val & 0x20) >> 1) | ((val & 0x40) >> 3);
        #endif
    }
}

float _calcSmartFreq(float reqFreq) {
    #if defined(__AVR__)
        if (reqFreq < 600.0) { _increment = 1; return reqFreq * 256.0; }
        else if (reqFreq < 1200.0) { _increment = 2; return reqFreq * 128.0; }
        else if (reqFreq < 2500.0) { _increment = 4; return reqFreq * 64.0; }
        else if (reqFreq < 5000.0) { _increment = 8; return reqFreq * 32.0; }
        else { _increment = 16; return reqFreq * 16.0; }
    #else
        if (reqFreq < 1500.0) { _increment = 1; return reqFreq * 256.0; }
        else if (reqFreq < 3000.0) { _increment = 2; return reqFreq * 128.0; }
        else if (reqFreq < 6000.0) { _increment = 4; return reqFreq * 64.0; }
        else if (reqFreq < 12000.0) { _increment = 16; return reqFreq * 16.0; }
        else { _increment = 32; return reqFreq * 8.0; }
    #endif
}

void _updateFrequency(float frequency) { 
    if (frequency <= 0) return;
    
    #if defined(__AVR__) 
        if (!_timerRunning) return; 
        
        float targetIntFreq = _calcSmartFreq(frequency);
        long cycles = (long)(16000000.0 / targetIntFreq); 
        uint8_t prescalerBits = 0;
        uint8_t ocrValue = 0;

        if (cycles < 256) { prescalerBits = (1 << CS20); ocrValue = cycles - 1; }
        else if (cycles / 8 < 256) { prescalerBits = (1 << CS21); ocrValue = (cycles / 8) - 1; }
        else if (cycles / 32 < 256) { prescalerBits = (1 << CS21) | (1 << CS20); ocrValue = (cycles / 32) - 1; }
        else if (cycles / 64 < 256) { prescalerBits = (1 << CS22); ocrValue = (cycles / 64) - 1; }
        else if (cycles / 128 < 256) { prescalerBits = (1 << CS22) | (1 << CS20); ocrValue = (cycles / 128) - 1; }
        else if (cycles / 256 < 256) { prescalerBits = (1 << CS22) | (1 << CS21); ocrValue = (cycles / 256) - 1; }
        else { prescalerBits = (1 << CS22) | (1 << CS21) | (1 << CS20); ocrValue = 255; }

        noInterrupts();
        OCR2A = ocrValue;
        TCCR2B = (TCCR2B & 0xF8) | prescalerBits; 
        interrupts();

    #elif defined(ARDUINO_ARCH_RENESAS)
        if (_r4TimerIndex < 0) return; 
        
        float targetFreq = _calcSmartFreq(frequency);
        
        _r4Timer.stop();
        _r4Timer.close();
        _r4Timer.begin(TIMER_MODE_PERIODIC, GPT_TIMER, _r4TimerIndex, targetFreq, 0.0f, r4Callback);
        _r4Timer.setup_overflow_irq(); 
        _r4Timer.open(); 
        _r4Timer.start();
    #endif
}

LadderDAC8::LadderDAC8() {}

void LadderDAC8::begin() {
    for (int i = 0; i < 8; i++) pinMode(dacPins[i], OUTPUT);
    pinMode(potPin, INPUT);
    pinMode(feedbackPin, INPUT);
	#ifdef analogReadResolution
		analogReadResolution(10);
	#endif
    
     #if defined(__AVR__)
        noInterrupts();
        TIMSK2 = 0; 
        TCCR2A = 0; 
        TCCR2B = 0; 
        TCNT2 = 0;  
        _timerRunning = false;
        interrupts();
    #endif

    clear(); 
 }

void LadderDAC8::clear() { write(0); }
void LadderDAC8::full() { write(255); }

void LadderDAC8::write(uint8_t value) {
    #if defined(__AVR__)
        if (_timerRunning) return; 
    #elif defined(ARDUINO_ARCH_RENESAS)
        if (_r4TimerIndex >= 0) return;
    #endif

    _lastValue = value;
    noInterrupts(); 

	#if defined(ARDUINO_AVR_UNO) || defined(ARDUINO_AVR_NANO)
        uint8_t portD_val = (value << 3) & 0xF8; 
        uint8_t portB_val = (value >> 5) & 0x07;

        PORTD = (PORTD & 0x07) | portD_val;
        PORTB = (PORTB & 0xF8) | portB_val;
        interrupts(); 
        return;
    #endif

    for (int i = 0; i < 8; i++) {
        digitalWrite(dacPins[i], (value >> i) & 1);
    }
    
    interrupts();
}

void LadderDAC8::writeBit(uint8_t bit, bool state) {
    if (bit > 7) return;
    if (state) _lastValue |= (1 << bit);
    else       _lastValue &= ~(1 << bit);
    digitalWrite(dacPins[bit], state ? HIGH : LOW);
}

void LadderDAC8::writeVoltage(float volts) {
    if (volts <= 0.0) { write(0); return; }
    if (volts >= _vref) { write(255); return; }
    write((uint8_t)((volts / _vref) * 255.0 + 0.5));
}

float LadderDAC8::getDacVoltage() { return (_lastValue / 255.0) * _vref; }

uint16_t LadderDAC8::_readStable(uint8_t pin) {
    uint32_t sum = 0;
    for (int i = 0; i < 8; i++) sum += analogRead(pin); 
    return (uint16_t)(sum / 8);
}
uint16_t LadderDAC8::readPotRaw() { return _readStable(potPin); }
uint8_t LadderDAC8::readPot() { return (readPotRaw() * 255UL) / 1023UL; }
uint16_t LadderDAC8::readFeedbackRaw() { return _readStable(feedbackPin); }
float LadderDAC8::readFeedbackVoltage() { return (readFeedbackRaw() * _vref) / 1023.0; }
uint8_t LadderDAC8::getPotPin() { return potPin; }
uint8_t LadderDAC8::getFeedbackPin() { return feedbackPin; }
void LadderDAC8::setReference(float v) { _vref = v; }
float LadderDAC8::getReference() { return _vref; }

void LadderDAC8::updateWaveFrequency(float frequency) {
    _updateFrequency(frequency);
}

#if defined(__AVR__) 

void _startTimerAVR(float frequency) {
    float targetIntFreq = _calcSmartFreq(frequency);
    long cycles = (long)(16000000.0 / targetIntFreq); 
    uint8_t prescalerBits = 0;
    uint8_t ocrValue = 0;

    if (cycles < 256) { prescalerBits = (1 << CS20); ocrValue = cycles - 1; }
    else if (cycles / 8 < 256) { prescalerBits = (1 << CS21); ocrValue = (cycles / 8) - 1; }
    else if (cycles / 32 < 256) { prescalerBits = (1 << CS21) | (1 << CS20); ocrValue = (cycles / 32) - 1; }
    else if (cycles / 64 < 256) { prescalerBits = (1 << CS22); ocrValue = (cycles / 64) - 1; }
    else if (cycles / 128 < 256) { prescalerBits = (1 << CS22) | (1 << CS20); ocrValue = (cycles / 128) - 1; }
    else if (cycles / 256 < 256) { prescalerBits = (1 << CS22) | (1 << CS21); ocrValue = (cycles / 256) - 1; }
    else { prescalerBits = (1 << CS22) | (1 << CS21) | (1 << CS20); ocrValue = 255; }

    if (_timerRunning) {
        return; 
    } else {
        noInterrupts();
        TCCR2A = 0; TCCR2B = 0; TCNT2  = 0;
        OCR2A = ocrValue;          
        TCCR2A |= (1 << WGM21);    
        TCCR2B |= prescalerBits;   
        TIMSK2 |= (1 << OCIE2A);   
        _timerRunning = true;
        _updatePortMaps(0); 
        interrupts();
    }
}

void LadderDAC8::waveSine(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(0);
    _timerDAC = this;
    if (_timerRunning) _updateFrequency(frequency);
    else _startTimerAVR(frequency);
}
void LadderDAC8::waveSquare(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(1);
    _timerDAC = this;
    if (_timerRunning) _updateFrequency(frequency);
    else _startTimerAVR(frequency);
}
void LadderDAC8::waveTriangle(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(2);
    _timerDAC = this;
    if (_timerRunning) _updateFrequency(frequency);
    else _startTimerAVR(frequency);
}
void LadderDAC8::waveSawtooth(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(3);
    _timerDAC = this;
    if (_timerRunning) _updateFrequency(frequency);
    else _startTimerAVR(frequency);
}
void LadderDAC8::waveStop() {
    TIMSK2 &= ~(1 << OCIE2A); 
    TCCR2B = 0; 
    _timerDAC = nullptr;
    _timerRunning = false;
}
ISR(TIMER2_COMPA_vect) {
    uint8_t idx = _sineIdx;
    _sineIdx += _increment;
    PORTD = (PORTD & 0x07) | _portD_Map[idx];
    PORTB = (PORTB & 0xF8) | _portB_Map[idx];
}

#elif defined(ARDUINO_ARCH_RENESAS)

void r4Callback(timer_callback_args_t *args) {
    uint8_t idx = _sineIdx;
    _sineIdx += _increment;
    #if defined(ARDUINO_MINIMA) || defined(ARDUINO_NANOR4)
        R_PORT1->PODR = (R_PORT1->PODR & ~0x10DC) | _r4_Port1_Map[idx];
        R_PORT3->PODR = (R_PORT3->PODR & ~0x0018) | _r4_Port3_Map[idx];
    #elif defined(ARDUINO_UNOWIFIR4)
        R_PORT1->PODR = (R_PORT1->PODR & ~0x18E8) | _r4_Port1_Map[idx];
        R_PORT3->PODR = (R_PORT3->PODR & ~0x0018) | _r4_Port3_Map[idx];
    #endif
}

void _startTimerR4(float frequency) {
    float targetFreq = _calcSmartFreq(frequency);
    
    if (_r4TimerIndex < 0) {
         uint8_t timer_type = GPT_TIMER; 
         _r4TimerIndex = FspTimer::get_available_timer(timer_type);
         if (_r4TimerIndex < 0) return; 

         _r4Timer.begin(TIMER_MODE_PERIODIC, GPT_TIMER, _r4TimerIndex, targetFreq, 0.0f, r4Callback);
         _r4Timer.setup_overflow_irq(); 
         _r4Timer.open(); 
         _r4Timer.start();
    } else {
         _r4TimerIndex = -1; 
         return;
    }
}

void LadderDAC8::waveSine(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(0);
    _timerDAC = this;
    if (_r4TimerIndex >= 0) _updateFrequency(frequency);
    else _startTimerR4(frequency);
}
void LadderDAC8::waveSquare(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(1);
    _timerDAC = this;
    if (_r4TimerIndex >= 0) _updateFrequency(frequency);
    else _startTimerR4(frequency);
}
void LadderDAC8::waveTriangle(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(2);
    _timerDAC = this;
    if (_r4TimerIndex >= 0) _updateFrequency(frequency);
    else _startTimerR4(frequency);
}
void LadderDAC8::waveSawtooth(float frequency) {
    if (frequency <= 0) return;
    _updatePortMaps(3);
    _timerDAC = this;
    if (_r4TimerIndex >= 0) _updateFrequency(frequency);
    else _startTimerR4(frequency);
}
void LadderDAC8::waveStop() {
    _r4Timer.stop(); 
    _r4Timer.close();
    _r4Timer.~FspTimer();
    new (&_r4Timer) FspTimer();
    _timerDAC = nullptr;
    _r4TimerIndex = -1; 
}
#endif